document.addEventListener('DOMContentLoaded', () => {
  // Show version from manifest
  const manifest = chrome.runtime.getManifest();
  const currentVersion = manifest.version;
  document.getElementById('version').textContent = `v${currentVersion}`;

  // Check for updates
  checkForUpdates(currentVersion);

  // Tab switching
  document.querySelectorAll('.tab').forEach(tab => {
    tab.addEventListener('click', () => {
      document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
      document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));

      tab.classList.add('active');
      document.getElementById(`${tab.dataset.tab}-tab`).classList.add('active');
    });
  });

  // Load all data
  loadAllData();

  // User controls
  document.getElementById('add-user-btn').addEventListener('click', addUser);
  document.getElementById('username-input').addEventListener('keypress', (e) => {
    if (e.key === 'Enter') addUser();
  });

  // Subforum controls
  document.getElementById('add-subforum-btn').addEventListener('click', addSubforum);
  document.getElementById('subforum-input').addEventListener('keypress', (e) => {
    if (e.key === 'Enter') addSubforum();
  });

  // Thread controls
  document.getElementById('add-thread-btn').addEventListener('click', addThread);
  document.getElementById('thread-input').addEventListener('keypress', (e) => {
    if (e.key === 'Enter') addThread();
  });

  function loadAllData() {
    chrome.runtime.sendMessage({ action: 'getBlockedData' }, (response) => {
      renderUsers(response.blockedUsers || {});
      renderSubforums(response.blockedSubforums || []);
      renderThreads(response.blockedThreads || []);
    });
  }

  function renderUsers(users) {
    const list = document.getElementById('users-list');
    list.innerHTML = '';

    const entries = Object.entries(users);

    if (entries.length === 0) {
      list.innerHTML = '<div class="empty">No blocked users. Right-click a username to block.</div>';
      return;
    }

    entries.sort((a, b) => a[0].toLowerCase().localeCompare(b[0].toLowerCase()));

    entries.forEach(([username, mode]) => {
      const item = document.createElement('div');
      item.className = 'list-item';
      item.innerHTML = `
        <span class="name">${escapeHtml(username)}</span>
        <span class="badge ${mode}">${mode}</span>
        <button class="toggle-btn" data-username="${escapeHtml(username)}" data-mode="${mode}">
          ${mode === 'disable' ? 'Terminate' : 'Disable'}
        </button>
        <button class="remove-btn" data-username="${escapeHtml(username)}">Remove</button>
      `;
      list.appendChild(item);
    });

    // Toggle mode handlers
    list.querySelectorAll('.toggle-btn').forEach(btn => {
      btn.addEventListener('click', () => {
        const newMode = btn.dataset.mode === 'disable' ? 'terminate' : 'disable';
        chrome.runtime.sendMessage({
          action: 'updateUserMode',
          username: btn.dataset.username,
          mode: newMode
        }, () => {
          loadAllData();
          notifyTabs();
        });
      });
    });

    // Remove handlers
    list.querySelectorAll('.remove-btn').forEach(btn => {
      btn.addEventListener('click', () => {
        chrome.runtime.sendMessage({
          action: 'unblockUser',
          username: btn.dataset.username
        }, () => {
          loadAllData();
          notifyTabs();
        });
      });
    });
  }

  function renderSubforums(subforums) {
    const list = document.getElementById('subforums-list');
    list.innerHTML = '';

    if (subforums.length === 0) {
      list.innerHTML = '<div class="empty">No blocked subforums. Right-click a forum link to block.</div>';
      return;
    }

    subforums.sort((a, b) => a.toLowerCase().localeCompare(b.toLowerCase()));

    subforums.forEach(subforum => {
      const item = document.createElement('div');
      item.className = 'list-item';
      item.innerHTML = `
        <span class="name">${escapeHtml(subforum)}</span>
        <button class="remove-btn" data-subforum="${escapeHtml(subforum)}">Remove</button>
      `;
      list.appendChild(item);
    });

    list.querySelectorAll('.remove-btn').forEach(btn => {
      btn.addEventListener('click', () => {
        chrome.runtime.sendMessage({
          action: 'unblockSubforum',
          subforum: btn.dataset.subforum
        }, () => {
          loadAllData();
          notifyTabs();
        });
      });
    });
  }

  function renderThreads(threads) {
    const list = document.getElementById('threads-list');
    list.innerHTML = '';

    if (threads.length === 0) {
      list.innerHTML = '<div class="empty">No blocked threads. Right-click a thread link to block.</div>';
      return;
    }

    // Normalize threads to objects and sort by title
    const normalizedThreads = threads.map(t => {
      if (typeof t === 'string') {
        return { id: t, title: `Thread #${t}` };
      }
      return t;
    });

    normalizedThreads.sort((a, b) => a.title.toLowerCase().localeCompare(b.title.toLowerCase()));

    normalizedThreads.forEach(thread => {
      const item = document.createElement('div');
      item.className = 'list-item';
      item.innerHTML = `
        <span class="name">${escapeHtml(thread.title)}</span>
        <button class="remove-btn" data-thread="${escapeHtml(thread.id)}">Remove</button>
      `;
      list.appendChild(item);
    });

    list.querySelectorAll('.remove-btn').forEach(btn => {
      btn.addEventListener('click', () => {
        chrome.runtime.sendMessage({
          action: 'unblockThread',
          thread: btn.dataset.thread
        }, () => {
          loadAllData();
          notifyTabs();
        });
      });
    });
  }

  function addUser() {
    const input = document.getElementById('username-input');
    const modeSelect = document.getElementById('mode-select');
    const username = input.value.trim();

    if (!username) return;

    chrome.runtime.sendMessage({
      action: 'blockUser',
      username: username,
      mode: modeSelect.value
    }, () => {
      input.value = '';
      loadAllData();
      notifyTabs();
    });
  }

  function addSubforum() {
    const input = document.getElementById('subforum-input');
    const subforum = input.value.trim();

    if (!subforum) return;

    chrome.runtime.sendMessage({
      action: 'blockSubforum',
      subforum: subforum
    }, () => {
      input.value = '';
      loadAllData();
      notifyTabs();
    });
  }

  function addThread() {
    const input = document.getElementById('thread-input');
    const thread = input.value.trim();

    if (!thread) return;

    chrome.runtime.sendMessage({
      action: 'blockThread',
      thread: thread
    }, () => {
      input.value = '';
      loadAllData();
      notifyTabs();
    });
  }

  function notifyTabs() {
    chrome.tabs.query({}, (tabs) => {
      tabs.forEach(tab => {
        chrome.tabs.sendMessage(tab.id, { action: 'refreshBlocking' }).catch(() => {});
      });
    });
  }

  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  function checkForUpdates(currentVersion) {
    fetch('https://webigniter.com/downloads/tango-down-version.txt?t=' + Date.now())
      .then(response => response.text())
      .then(latestVersion => {
        latestVersion = latestVersion.trim();
        if (latestVersion && compareVersions(latestVersion, currentVersion) > 0) {
          showUpdateNotice(latestVersion);
        }
      })
      .catch(() => {});
  }

  function compareVersions(a, b) {
    const partsA = a.split('.').map(Number);
    const partsB = b.split('.').map(Number);
    for (let i = 0; i < Math.max(partsA.length, partsB.length); i++) {
      const numA = partsA[i] || 0;
      const numB = partsB[i] || 0;
      if (numA > numB) return 1;
      if (numA < numB) return -1;
    }
    return 0;
  }

  function showUpdateNotice(newVersion) {
    const notice = document.createElement('div');
    notice.className = 'update-notice';
    notice.innerHTML = `
      <span>Update available: v${newVersion}</span>
      <a href="https://webigniter.com/tango-down" target="_blank">Download</a>
    `;
    document.body.insertBefore(notice, document.body.firstChild);
  }
});
