// Tango Down - vBulletin User Blocker Content Script v1.5.0

let blockedUsers = {};
let blockedSubforums = [];
let blockedThreads = [];
let lastRightClickedElement = null;

document.addEventListener("contextmenu", (e) => {
  lastRightClickedElement = e.target;
});

async function init() {
  await loadBlockedData();
  hideBlockedContent();
  observeNewContent();
}

async function loadBlockedData() {
  return new Promise((resolve) => {
    chrome.storage.sync.get(["blockedUsers", "blockedSubforums", "blockedThreads"], (result) => {
      const users = result.blockedUsers || {};
      if (Array.isArray(users)) {
        blockedUsers = {};
        users.forEach(u => blockedUsers[u.toLowerCase()] = 'disable');
      } else {
        blockedUsers = {};
        Object.keys(users).forEach(u => blockedUsers[u.toLowerCase()] = users[u]);
      }
      blockedSubforums = result.blockedSubforums || [];
      blockedThreads = result.blockedThreads || [];
      console.log('[Tango Down] Loaded:', { users: Object.keys(blockedUsers).length, threads: blockedThreads.length, subforums: blockedSubforums.length });
      resolve();
    });
  });
}

function hideBlockedContent() {
  hideBlockedPosts();
  hideBlockedQuotes();
  hideBlockedSubforumLinks();
  hideBlockedThreadLinks();
}

function hideBlockedPosts() {
  if (Object.keys(blockedUsers).length === 0) return;

  const usernameSelectors = [
    'a.bigusername',
    'a[href*="member.php?u="]',
    'a[href*="member.php/"]',
    '.bigusername',
    '.username',
    'a.username'
  ];

  document.querySelectorAll(usernameSelectors.join(', ')).forEach(usernameEl => {
    const username = extractUsername(usernameEl);
    if (!isBlocked(username)) return;

    let container = null;
    container = usernameEl.closest('table[id^="post"]');
    if (!container) container = usernameEl.closest('tr[id^="post"]');
    if (!container) container = usernameEl.closest('.postbit, .postbitlegacy, .postcontainer, div[id^="post_"]');

    if (!container) {
      let el = usernameEl.parentElement;
      while (el && el !== document.body) {
        if (el.tagName === 'TABLE' && el.querySelector('div[id^="post_message"]')) {
          container = el;
          break;
        }
        el = el.parentElement;
      }
    }

    if (!container) {
      container = usernameEl.closest('tbody');
      if (container && !container.querySelector('div[id^="post_message"]')) {
        container = null;
      }
    }

    if (container && !container.classList.contains('td-hidden')) {
      hideElement(container, username, 'post');
    }
  });
}

function hideBlockedQuotes() {
  if (Object.keys(blockedUsers).length === 0) return;

  // Method 1: Find standard quote containers
  const quoteSelectors = [
    '.quote_container',
    '.bbcode_quote',
    '.quote',
    '.b-quote',
    '.bbcode-quote',
    'blockquote',
    'div[style*="margin:20px"]',
    'td.alt2[style*="border"]'
  ];

  const quotes = document.querySelectorAll(quoteSelectors.join(', '));

  quotes.forEach(quote => {
    if (quote.classList.contains('td-quote-processed')) return;
    quote.classList.add('td-quote-processed');

    const attributionText = getQuoteAttribution(quote);

    if (attributionText) {
      for (const blockedUser of Object.keys(blockedUsers)) {
        if (attributionText.toLowerCase().includes(blockedUser)) {
          hideElement(quote, blockedUser, 'quote');
          break;
        }
      }
    }
  });

  // Method 2: Find elements containing "Originally Posted by" and check if blocked user is nearby
  // Username is often in a child element: "Originally Posted by <strong>Username</strong>"
  const allElements = document.querySelectorAll('*');

  allElements.forEach(el => {
    if (el.classList?.contains('td-quote-processed')) return;
    if (el.classList?.contains('td-hidden')) return;

    // Check direct text content (not children)
    const directText = Array.from(el.childNodes)
      .filter(n => n.nodeType === Node.TEXT_NODE)
      .map(n => n.textContent)
      .join('');

    if (!directText.toLowerCase().includes('posted by')) return;

    // Found "Posted by" - now check if a blocked username is in this element or its children
    const fullText = el.textContent || '';

    for (const blockedUser of Object.keys(blockedUsers)) {
      if (!fullText.toLowerCase().includes(blockedUser.toLowerCase())) continue;

      console.log('[Tango Down] Found quote attribution for:', blockedUser, 'in element:', el.tagName);

      // Walk up to find the quote container
      let container = el;
      let quoteContainer = null;

      for (let i = 0; i < 12 && container; i++) {
        // Stop conditions
        if (container.id?.startsWith('post_message')) break;
        if (container.id?.startsWith('post') && container.tagName === 'TABLE') break;

        const tag = container.tagName;
        const style = container.getAttribute?.('style') || '';

        // Look for quote-like containers
        if (tag === 'TABLE' ||
            tag === 'BLOCKQUOTE' ||
            (tag === 'DIV' && style.includes('margin')) ||
            (tag === 'TD' && style.includes('border'))) {
          quoteContainer = container;
        }

        container = container.parentElement;
      }

      if (quoteContainer) {
        // Verify we're inside a post message (not the post itself)
        const postMessage = quoteContainer.closest('div[id^="post_message"]');
        if (postMessage && quoteContainer !== postMessage) {
          quoteContainer.classList.add('td-quote-processed');
          hideElement(quoteContainer, blockedUser, 'quote');
          console.log('[Tango Down] Hid quote container:', quoteContainer.tagName, 'from:', blockedUser);
          return;
        } else {
          console.log('[Tango Down] Skipped - not inside post_message or is post_message itself');
        }
      } else {
        console.log('[Tango Down] No suitable quote container found');
      }
    }
  });
}

// Extract thread ID from various URL formats
function extractThreadId(href) {
  if (!href) return null;

  // Format 1: showthread.php?t=123456
  let match = href.match(/showthread\.php\?.*t=(\d+)/);
  if (match) return match[1];

  // Format 2: showthread.php/123456-Thread-Title
  match = href.match(/showthread\.php\/(\d+)/);
  if (match) return match[1];

  // Format 3: GFY SEO URLs - /forum-name/123456-thread-title.html
  match = href.match(/\/(\d+)-[^\/]+\.html/);
  if (match) return match[1];

  // Format 4: Just a number followed by hyphen in path (5+ digits to avoid false matches)
  match = href.match(/\/(\d{5,})-/);
  if (match) return match[1];

  return null;
}

// Extract forum identifier from URL
function extractForumId(href) {
  if (!href) return null;

  // Format 1: forumdisplay.php?f=123
  let match = href.match(/forumdisplay\.php\?.*f=(\d+)/);
  if (match) return match[1];

  // Format 2: forumdisplay.php/123-Forum-Name
  match = href.match(/forumdisplay\.php\/(\d+)/);
  if (match) return match[1];

  // Format 3: GFY SEO URLs - /forum-name/
  try {
    const url = new URL(href, window.location.origin);
    const path = url.pathname;

    // Skip if this looks like a thread
    if (/\/\d+-[^\/]+\.html/.test(path)) return null;

    // Get the forum slug
    match = path.match(/^\/([a-z0-9-]+)\/?$/i);
    if (match) return match[1];

    match = path.match(/^\/[^\/]+\/([a-z0-9-]+)\/?$/i);
    if (match) return match[1];
  } catch (e) {}

  return null;
}

// Check if a link is a thread link
function isThreadLink(href) {
  if (!href) return false;
  if (href.includes('showthread.php')) return true;
  if (/\/\d+-[^\/]+\.html/.test(href)) return true;
  return false;
}

// Check if a link is a forum link
function isForumLink(href) {
  if (!href) return false;
  if (href.includes('forumdisplay.php')) return true;
  if (isThreadLink(href)) return false;
  try {
    const url = new URL(href, window.location.origin);
    const path = url.pathname;
    if (/^\/[a-z0-9-]+\/?$/i.test(path) && path !== '/') return true;
  } catch (e) {}
  return false;
}

function hideBlockedSubforumLinks() {
  if (blockedSubforums.length === 0) return;

  // Find all links that could be forum links
  document.querySelectorAll('a[href*="forumdisplay.php"], a[href]').forEach(link => {
    const href = link.href || link.getAttribute('href') || '';

    if (!isForumLink(href)) return;

    const forumId = extractForumId(href);
    if (!forumId) return;

    if (blockedSubforums.includes(forumId)) {
      let row = findListingRow(link);
      if (row && !row.classList.contains('td-hidden')) {
        row.style.display = 'none';
        row.classList.add('td-hidden');
        console.log('[Tango Down] Hid subforum:', forumId);
      }
    }
  });
}

function hideBlockedThreadLinks() {
  if (blockedThreads.length === 0) return;

  // Find all links that could be thread links
  document.querySelectorAll('a[href*="showthread.php"], a[href*=".html"]').forEach(link => {
    const href = link.href || link.getAttribute('href') || '';

    if (!isThreadLink(href)) return;

    const threadId = extractThreadId(href);
    if (!threadId) return;

    // Check if blocked (handle both old string format and new object format)
    const isBlocked = blockedThreads.some(t =>
      (typeof t === 'string' && t === threadId) ||
      (typeof t === 'object' && t.id === threadId)
    );

    if (isBlocked) {
      let row = findListingRow(link);
      if (row && !row.classList.contains('td-hidden')) {
        row.style.display = 'none';
        row.classList.add('td-hidden');
        console.log('[Tango Down] Hid thread:', threadId);
      }
    }
  });
}

// Find the listing row for a thread/forum link
function findListingRow(link) {
  // Strategy 1: Look for tr with multiple cells
  let row = link.closest('tr');
  if (row) {
    const cells = row.querySelectorAll('td');
    if (cells.length >= 2) return row;
  }

  // Strategy 2: Look for tbody that contains the thread/forum
  row = link.closest('tbody');
  if (row && row.parentElement && row.parentElement.tagName === 'TABLE') {
    // Make sure it's not the main table tbody
    const tbodies = row.parentElement.querySelectorAll('tbody');
    if (tbodies.length > 1) return row;
  }

  // Strategy 3: li element
  row = link.closest('li');
  if (row) return row;

  // Strategy 4: Thread-specific divs
  row = link.closest('div.threadbit, div.thread, div[id^="thread_"], div[id*="thread"]');
  if (row) return row;

  // Strategy 5: Look for parent tr even if it has few cells
  row = link.closest('tr');
  if (row) return row;

  // Strategy 6: For GFY - find a table that contains this link and hide the whole table if it's small
  let table = link.closest('table');
  if (table) {
    const rows = table.querySelectorAll('tr');
    if (rows.length <= 3) return table;
  }

  // Fallback: hide the link's parent element
  return link.parentElement;
}

function getQuoteAttribution(quoteEl) {
  const attrSelectors = ['.quote_author', '.bbcode_postedby', '.bb_quote_author', '.quotetitle', '.attribution', 'cite', 'strong', 'b'];

  for (const selector of attrSelectors) {
    const attrEl = quoteEl.querySelector(selector);
    if (attrEl && attrEl.textContent.toLowerCase().includes('posted by')) {
      return attrEl.textContent;
    }
    if (attrEl && attrEl.closest('.bbcode_postedby, .quote_container')) {
      return attrEl.textContent;
    }
  }

  const quoteText = quoteEl.textContent || '';
  const match = quoteText.match(/(?:Originally Posted by|Quote from|Posted by)\s*([^\n\r\d]+)/i);
  if (match) return match[1];

  return quoteEl.getAttribute('data-author') || quoteEl.getAttribute('data-username') || '';
}

function extractUsername(element) {
  let username = element.textContent || element.innerText || '';
  username = username.replace(/\s*(is online|is offline|View Profile|View Forum Posts|Send a private message)\s*/gi, '').replace(/^\s*@/, '').trim();
  return username;
}

function isBlocked(username) {
  if (!username) return false;
  return username.toLowerCase() in blockedUsers;
}

function getBlockMode(username) {
  if (!username) return 'disable';
  return blockedUsers[username.toLowerCase()] || 'disable';
}

function hideElement(element, username, type) {
  if (element.classList.contains('td-hidden')) return;

  const mode = getBlockMode(username);
  element.classList.add('td-hidden');

  if (mode === 'terminate') {
    element.style.display = 'none';
    return;
  }

  element.style.display = 'none';

  const placeholder = document.createElement('div');
  placeholder.className = 'td-blocked-placeholder';
  placeholder.innerHTML = `
    <span class="td-blocked-icon">🚫</span>
    <span class="td-blocked-text">Blocked ${type} from <strong>${escapeHtml(username)}</strong></span>
    <button class="td-show-anyway">Show anyway</button>
  `;

  element.parentNode.insertBefore(placeholder, element);

  placeholder.querySelector('.td-show-anyway').addEventListener('click', () => {
    element.style.display = '';
    element.classList.remove('td-hidden');
    element.classList.add('td-revealed');
    placeholder.remove();
  });
}

function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

function observeNewContent() {
  const observer = new MutationObserver((mutations) => {
    let shouldCheck = false;
    for (const mutation of mutations) {
      if (mutation.addedNodes.length > 0) {
        shouldCheck = true;
        break;
      }
    }
    if (shouldCheck) {
      clearTimeout(window.tdDebounce);
      window.tdDebounce = setTimeout(() => hideBlockedContent(), 100);
    }
  });

  observer.observe(document.body, { childList: true, subtree: true });
}

function showNotification(message) {
  const notif = document.createElement('div');
  notif.style.cssText = 'position:fixed;top:20px;right:20px;background:#10b981;color:#fff;padding:12px 20px;border-radius:8px;font-family:sans-serif;font-size:14px;z-index:99999;box-shadow:0 4px 12px rgba(0,0,0,0.3);';
  notif.textContent = message;
  document.body.appendChild(notif);
  setTimeout(() => notif.remove(), 3000);
}

function fullRefresh() {
  document.querySelectorAll('.td-blocked-placeholder').forEach(el => el.remove());
  document.querySelectorAll('.td-hidden, .td-processed, .td-quote-processed').forEach(el => {
    el.style.display = '';
    el.classList.remove('td-hidden', 'td-processed', 'td-quote-processed');
  });
  hideBlockedContent();
}

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === "getClickedUsername") {
    const username = detectUsernameFromClick(message);
    sendResponse({ username });
    return false;
  }

  if (message.action === "getLinkTitle") {
    let title = null;
    if (lastRightClickedElement) {
      const link = lastRightClickedElement.closest('a') ||
                   (lastRightClickedElement.tagName === 'A' ? lastRightClickedElement : null);
      if (link) {
        title = link.textContent.trim();
      }
    }
    if (!title && message.linkUrl) {
      const links = document.querySelectorAll('a');
      for (const link of links) {
        const href = link.href || '';
        if (href === message.linkUrl || href.includes(message.linkUrl) || message.linkUrl.includes(href)) {
          const text = link.textContent.trim();
          if (text && text.length > 3) {
            title = text;
            break;
          }
        }
      }
    }
    sendResponse({ title });
    return false;
  }

  if (message.action === "refreshBlocking") {
    loadBlockedData().then(() => {
      fullRefresh();
      sendResponse({ success: true });
    });
    return true; // async
  }

  if (message.action === "showNotification") {
    showNotification(message.message);
    sendResponse({ success: true });
    return false;
  }

  return false;
});

function detectUsernameFromClick(message) {
  if (message.selectionText) return message.selectionText.trim();

  if (lastRightClickedElement) {
    const clickedEl = lastRightClickedElement;

    const usernameLink = clickedEl.closest('a.username, a.bigusername, a[href*="member.php?u="], a[href*="/members/"]');
    if (usernameLink) return extractUsername(usernameLink);

    const userInfo = clickedEl.closest('.userinfo, .poster_info, .postbit, td.alt2');
    if (userInfo) {
      const usernameEl = userInfo.querySelector('.username, .bigusername, a[href*="member.php?u="]');
      if (usernameEl) return extractUsername(usernameEl);
    }

    if (clickedEl.tagName === 'A' && clickedEl.textContent.trim()) {
      return clickedEl.textContent.trim();
    }
  }

  return null;
}

const style = document.createElement('style');
style.textContent = `
  .td-blocked-placeholder {
    background: #1a1a1a;
    border: 1px solid #333;
    border-radius: 4px;
    padding: 12px 16px;
    margin: 8px 0;
    display: flex;
    align-items: center;
    gap: 12px;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    font-size: 14px;
    color: #888;
  }
  .td-blocked-icon { font-size: 18px; }
  .td-blocked-text { flex: 1; }
  .td-blocked-text strong { color: #fff; }
  .td-show-anyway {
    background: #333;
    border: 1px solid #444;
    color: #888;
    padding: 6px 12px;
    border-radius: 4px;
    cursor: pointer;
    font-size: 12px;
    transition: all 0.2s;
  }
  .td-show-anyway:hover { background: #444; color: #fff; }
  .td-revealed { opacity: 0.6; border-left: 3px solid #ef4444 !important; }
`;
document.head.appendChild(style);

init();
