// Tango Down - Background Script v1.5.0

// Create context menus on install
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: "blockUserDisable",
    title: "Block user (Disable - can reveal)",
    contexts: ["selection", "link"]
  });

  chrome.contextMenus.create({
    id: "blockUserTerminate",
    title: "Block user (Terminate - no trace)",
    contexts: ["selection", "link"]
  });

  chrome.contextMenus.create({
    id: "separator1",
    type: "separator",
    contexts: ["selection", "link"]
  });

  chrome.contextMenus.create({
    id: "blockThread",
    title: "Block this thread",
    contexts: ["link"]
  });

  chrome.contextMenus.create({
    id: "blockSubforum",
    title: "Block this subforum",
    contexts: ["link"]
  });
});

// Extract thread ID from various URL formats
function extractThreadId(url) {
  if (!url) return null;

  // Format 1: showthread.php?t=123456
  let match = url.match(/showthread\.php\?.*t=(\d+)/);
  if (match) return match[1];

  // Format 2: showthread.php/123456-Thread-Title
  match = url.match(/showthread\.php\/(\d+)/);
  if (match) return match[1];

  // Format 3: GFY SEO URLs - /forum-name/123456-thread-title.html
  match = url.match(/\/(\d+)-[^\/]+\.html/);
  if (match) return match[1];

  // Format 4: Just a number followed by hyphen in path
  match = url.match(/\/(\d{5,})-/);
  if (match) return match[1];

  return null;
}

// Extract thread title from SEO-friendly URL
function extractTitleFromUrl(url) {
  if (!url) return null;

  // GFY SEO format: /forum-name/123456-thread-title-here.html
  let match = url.match(/\/\d+-([^\/]+)\.html/);
  if (match) {
    // Convert slug to title: "thread-title-here" -> "Thread Title Here"
    return match[1]
      .replace(/-/g, ' ')
      .replace(/\b\w/g, c => c.toUpperCase())
      .substring(0, 50);
  }

  // showthread.php/123456-Thread-Title format
  match = url.match(/showthread\.php\/\d+-(.+?)(?:\?|$)/);
  if (match) {
    return match[1]
      .replace(/-/g, ' ')
      .replace(/\b\w/g, c => c.toUpperCase())
      .substring(0, 50);
  }

  return null;
}

// Extract forum identifier from URL (ID or slug)
function extractForumId(url) {
  if (!url) return null;

  // Format 1: forumdisplay.php?f=123
  let match = url.match(/forumdisplay\.php\?.*f=(\d+)/);
  if (match) return match[1];

  // Format 2: forumdisplay.php/123-Forum-Name
  match = url.match(/forumdisplay\.php\/(\d+)/);
  if (match) return match[1];

  // Format 3: GFY SEO URLs - /forum-name/ (use the slug as ID)
  // Match path that doesn't contain numbers followed by .html (those are threads)
  try {
    const urlObj = new URL(url);
    const path = urlObj.pathname;

    // Skip if this looks like a thread (has /123456-title.html pattern)
    if (/\/\d+-[^\/]+\.html/.test(path)) return null;

    // Get the forum slug from path like /forum-name/ or /forum-name
    match = path.match(/^\/([a-z0-9-]+)\/?$/i);
    if (match) return match[1];

    // Also match /category/forum-name/
    match = path.match(/^\/[^\/]+\/([a-z0-9-]+)\/?$/i);
    if (match) return match[1];
  } catch (e) {}

  return null;
}

// Check if URL is a thread link
function isThreadUrl(url) {
  if (!url) return false;
  if (url.includes('showthread.php')) return true;
  // GFY SEO format: /forum-name/123456-title.html
  if (/\/\d+-[^\/]+\.html/.test(url)) return true;
  return false;
}

// Check if URL is a forum link
function isForumUrl(url) {
  if (!url) return false;
  if (url.includes('forumdisplay.php')) return true;
  // GFY SEO format: /forum-name/ (but NOT thread URLs)
  if (isThreadUrl(url)) return false;
  try {
    const urlObj = new URL(url);
    const path = urlObj.pathname;
    // Match /something/ or /something (but not homepage or files)
    if (/^\/[a-z0-9-]+\/?$/i.test(path) && path !== '/') return true;
  } catch (e) {}
  return false;
}

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener((info, tab) => {
  if (info.menuItemId === "blockUserDisable" || info.menuItemId === "blockUserTerminate") {
    const mode = info.menuItemId === "blockUserDisable" ? "disable" : "terminate";

    chrome.tabs.sendMessage(tab.id, {
      action: "getClickedUsername",
      selectionText: info.selectionText,
      linkUrl: info.linkUrl
    }, (response) => {
      const username = response?.username || info.selectionText?.trim();
      if (username) {
        blockUser(username, mode).then(() => {
          chrome.tabs.sendMessage(tab.id, { action: "refreshBlocking" });
          chrome.tabs.sendMessage(tab.id, {
            action: "showNotification",
            message: `User "${username}" blocked (${mode} mode)!`
          });
        });
      }
    });
  }

  if (info.menuItemId === "blockThread") {
    const url = info.linkUrl || '';
    console.log('[Tango Down] Block thread clicked, URL:', url);

    if (isThreadUrl(url)) {
      const threadId = extractThreadId(url);
      if (threadId) {
        // Try to get title from URL first (most reliable for SEO URLs)
        const urlTitle = extractTitleFromUrl(url);

        // Also try content script as backup
        chrome.tabs.sendMessage(tab.id, {
          action: "getLinkTitle",
          linkUrl: url
        }, (response) => {
          // Prefer content script title, fall back to URL title, then ID
          const title = response?.title || urlTitle || `Thread #${threadId}`;
          console.log('[Tango Down] Blocking thread:', threadId, 'Title:', title);
          blockThread(threadId, title).then(() => {
            chrome.tabs.sendMessage(tab.id, { action: "refreshBlocking" });
            chrome.tabs.sendMessage(tab.id, {
              action: "showNotification",
              message: `Thread blocked: ${title.substring(0, 30)}${title.length > 30 ? '...' : ''}`
            });
          });
        });
      } else {
        console.log('[Tango Down] Could not extract thread ID from:', url);
        chrome.tabs.sendMessage(tab.id, {
          action: "showNotification",
          message: `Could not identify thread from URL`
        });
      }
    } else {
      console.log('[Tango Down] Not a thread URL:', url);
      chrome.tabs.sendMessage(tab.id, {
        action: "showNotification",
        message: `Not a thread link`
      });
    }
  }

  if (info.menuItemId === "blockSubforum") {
    const url = info.linkUrl || '';
    console.log('[Tango Down] Block subforum clicked, URL:', url);

    if (isForumUrl(url)) {
      const forumId = extractForumId(url);
      if (forumId) {
        console.log('[Tango Down] Blocking subforum:', forumId);
        blockSubforum(forumId).then(() => {
          chrome.tabs.sendMessage(tab.id, { action: "refreshBlocking" });
          chrome.tabs.sendMessage(tab.id, {
            action: "showNotification",
            message: `Subforum "${forumId}" blocked!`
          });
        });
      } else {
        console.log('[Tango Down] Could not extract forum ID from:', url);
        chrome.tabs.sendMessage(tab.id, {
          action: "showNotification",
          message: `Could not identify subforum from URL`
        });
      }
    } else {
      console.log('[Tango Down] Not a forum URL:', url);
      chrome.tabs.sendMessage(tab.id, {
        action: "showNotification",
        message: `Not a subforum link`
      });
    }
  }
});

// Listen for messages from popup/content
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  // Handle async actions
  const handleAsync = async () => {
    switch (message.action) {
      case "blockUser":
        await blockUser(message.username, message.mode || "disable");
        if (sender.tab) {
          chrome.tabs.sendMessage(sender.tab.id, { action: "refreshBlocking" }).catch(() => {});
        }
        return { success: true };

      case "getBlockedData":
        const result = await chrome.storage.sync.get(["blockedUsers", "blockedSubforums", "blockedThreads"]);
        return {
          blockedUsers: result.blockedUsers || {},
          blockedSubforums: result.blockedSubforums || [],
          blockedThreads: result.blockedThreads || []
        };

      case "unblockUser":
        await unblockUser(message.username);
        return { success: true };

      case "unblockSubforum":
        await unblockSubforum(message.subforum);
        return { success: true };

      case "unblockThread":
        await unblockThread(message.thread);
        return { success: true };

      case "updateUserMode":
        await updateUserMode(message.username, message.mode);
        return { success: true };

      case "blockSubforum":
        await blockSubforum(message.subforum);
        return { success: true };

      case "blockThread":
        await blockThread(message.thread);
        return { success: true };

      default:
        return null;
    }
  };

  // Only return true (async) if we handle this action
  if (message.action) {
    handleAsync().then(response => {
      sendResponse(response || { success: true });
    }).catch(err => {
      console.error('[Tango Down] Error:', err);
      sendResponse({ success: false, error: err.message });
    });
    return true;
  }

  return false;
});

async function blockUser(username, mode = "disable") {
  const result = await chrome.storage.sync.get(["blockedUsers"]);
  let blockedUsers = result.blockedUsers || {};

  if (Array.isArray(blockedUsers)) {
    const old = blockedUsers;
    blockedUsers = {};
    old.forEach(u => blockedUsers[u] = "disable");
  }

  blockedUsers[username.trim()] = mode;
  await chrome.storage.sync.set({ blockedUsers });
  console.log('[Tango Down] Blocked user:', username, 'mode:', mode);
}

async function unblockUser(username) {
  const result = await chrome.storage.sync.get(["blockedUsers"]);
  let blockedUsers = result.blockedUsers || {};

  const key = Object.keys(blockedUsers).find(k => k.toLowerCase() === username.toLowerCase());
  if (key) delete blockedUsers[key];

  await chrome.storage.sync.set({ blockedUsers });
}

async function updateUserMode(username, mode) {
  const result = await chrome.storage.sync.get(["blockedUsers"]);
  let blockedUsers = result.blockedUsers || {};

  const key = Object.keys(blockedUsers).find(k => k.toLowerCase() === username.toLowerCase());
  if (key) blockedUsers[key] = mode;

  await chrome.storage.sync.set({ blockedUsers });
}

async function blockThread(threadId, title = null) {
  const result = await chrome.storage.sync.get(["blockedThreads"]);
  let blockedThreads = result.blockedThreads || [];

  // Check if thread is already blocked (handle both old string format and new object format)
  const alreadyBlocked = blockedThreads.some(t =>
    (typeof t === 'string' && t === threadId) ||
    (typeof t === 'object' && t.id === threadId)
  );

  if (!alreadyBlocked) {
    // Store as object with id and title
    const threadEntry = {
      id: threadId,
      title: title ? title.substring(0, 50) : `Thread #${threadId}`
    };
    blockedThreads.push(threadEntry);
    await chrome.storage.sync.set({ blockedThreads });
    console.log('[Tango Down] Thread added to blocklist:', threadEntry, 'Total:', blockedThreads.length);
  }
}

async function unblockThread(threadId) {
  const result = await chrome.storage.sync.get(["blockedThreads"]);
  let blockedThreads = result.blockedThreads || [];
  // Handle both old string format and new object format
  blockedThreads = blockedThreads.filter(t =>
    (typeof t === 'string' && t !== threadId) ||
    (typeof t === 'object' && t.id !== threadId)
  );
  await chrome.storage.sync.set({ blockedThreads });
}

async function blockSubforum(forumId) {
  const result = await chrome.storage.sync.get(["blockedSubforums"]);
  const blockedSubforums = result.blockedSubforums || [];

  if (!blockedSubforums.includes(forumId)) {
    blockedSubforums.push(forumId);
    await chrome.storage.sync.set({ blockedSubforums });
    console.log('[Tango Down] Subforum added to blocklist:', forumId, 'Total:', blockedSubforums.length);
  }
}

async function unblockSubforum(forumId) {
  const result = await chrome.storage.sync.get(["blockedSubforums"]);
  let blockedSubforums = result.blockedSubforums || [];
  blockedSubforums = blockedSubforums.filter(f => f !== forumId);
  await chrome.storage.sync.set({ blockedSubforums });
}
